<?php
/**
 * Binance P2P Settings
 * Configure Bridge account, Binance API, and automation settings
 */

require_once '../includes/config.php';
require_once '../includes/functions.php';
require_once '../includes/binance-settings.php';
require_once '../includes/bridge-api.php';

require_login();

$settings = new BinanceP2PSettings($pdo);
$message = '';
$messageType = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['save_binance_credentials'])) {
        $apiKey = $_POST['binance_api_key'] ?? '';
        $secretKey = $_POST['binance_secret_key'] ?? '';
        
        if (!empty($apiKey) && !empty($secretKey)) {
            $settings->setBinanceCredentials($apiKey, $secretKey);
            $message = "Binance API credentials saved successfully!";
            $messageType = 'success';
        } else {
            $message = "Please provide both API key and secret key.";
            $messageType = 'danger';
        }
    }
    
    if (isset($_POST['save_bridge_account'])) {
        $customerId = $_POST['bridge_customer_id'] ?? '';
        $externalAccountId = $_POST['bridge_external_account_id'] ?? '';
        
        if (!empty($customerId) && !empty($externalAccountId)) {
            $settings->setBridgeCustomerId($customerId);
            $settings->setBridgeExternalAccountId($externalAccountId);
            $message = "Bridge account saved successfully!";
            $messageType = 'success';
        } else {
            $message = "Please provide both Bridge customer ID and external account ID.";
            $messageType = 'danger';
        }
    }
    
    if (isset($_POST['save_automation_settings'])) {
        $autoPayment = isset($_POST['auto_payment_enabled']) ? 1 : 0;
        $lowBalanceThreshold = floatval($_POST['low_balance_threshold'] ?? 100);
        
        $settings->setAutoPaymentEnabled($autoPayment);
        $settings->set('low_balance_threshold', $lowBalanceThreshold);
        
        $message = "Automation settings saved successfully!";
        $messageType = 'success';
    }
    
    if (isset($_POST['save_wallet_settings'])) {
        $walletAddress = trim($_POST['wallet_address'] ?? '');
        $walletPrivateKey = trim($_POST['wallet_private_key'] ?? '');
        $withdrawalWalletAddress = trim($_POST['withdrawal_wallet_address'] ?? '');
        
        if (!empty($walletAddress) && !empty($walletPrivateKey)) {
            // Encrypt private key before storing
            require_once '../includes/functions.php';
            $encryptedKey = encrypt_data($walletPrivateKey, ENCRYPTION_KEY);
            
            $settings->set('wallet_address', $walletAddress);
            $settings->set('wallet_private_key', $encryptedKey);
            
            $message = "Wallet settings saved successfully!";
            $messageType = 'success';
        } else {
            $message = "Please provide both wallet address and private key.";
            $messageType = 'danger';
        }
        
        // Save withdrawal wallet address separately (doesn't require private key)
        if (!empty($withdrawalWalletAddress)) {
            $settings->set('withdrawal_wallet_address', $withdrawalWalletAddress);
            if ($messageType === 'success') {
                $message .= " Withdrawal wallet address saved.";
            }
        }
    }
    
    if (isset($_POST['save_translation_settings'])) {
        $systranApiKey = trim($_POST['systran_api_key'] ?? '');
        
        if (!empty($systranApiKey)) {
            $settings->set('systran_api_key', $systranApiKey);
            $message = "Translation settings saved successfully!";
            $messageType = 'success';
        } else {
            $settings->set('systran_api_key', '');
            $message = "SYSTRAN API key cleared (name translation will be disabled).";
            $messageType = 'info';
        }
    }
}

// Get current settings
$binanceApiKey = $settings->getBinanceApiKey();
$binanceSecretKey = $settings->getBinanceSecretKey();
$bridgeCustomerId = $settings->getBridgeCustomerId();
$bridgeExternalAccountId = $settings->getBridgeExternalAccountId();
$autoPaymentEnabled = $settings->isAutoPaymentEnabled();
$lowBalanceThreshold = $settings->getLowBalanceThreshold();
$walletAddress = $settings->get('wallet_address');
$walletPrivateKeyEncrypted = $settings->get('wallet_private_key');
$withdrawalWalletAddress = $settings->get('withdrawal_wallet_address');
$systranApiKey = $settings->get('systran_api_key');

// Get Bridge account details if available
$bridgeAccountDetails = null;
if ($bridgeCustomerId && $bridgeExternalAccountId) {
    try {
        $bridge = new BridgeAPI(BRIDGE_API_KEY);
        $bridgeAccountDetails = $bridge->getExternalAccountDetails($bridgeCustomerId, $bridgeExternalAccountId);
    } catch (Exception $e) {
        error_log("Error getting Bridge account details: " . $e->getMessage());
    }
}

$page_title = "Settings";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f5f5f5;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        .navbar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-dark">
        <div class="container-fluid">
            <span class="navbar-brand mb-0 h1">
                <i class="bi bi-gear"></i> Settings
            </span>
            <a href="orders.php" class="btn btn-light btn-sm">
                <i class="bi bi-arrow-left"></i> Back to Orders
            </a>
        </div>
    </nav>

    <div class="container-fluid mt-4">
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                <?php echo sanitize($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Binance API Credentials -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-key"></i> Binance API Credentials</h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">Binance API Key</label>
                        <input type="text" name="binance_api_key" class="form-control" 
                               value="<?php echo sanitize($binanceApiKey ?? ''); ?>" 
                               placeholder="Enter your Binance API key">
                        <small class="form-text text-muted">Get this from Binance API Management</small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Binance Secret Key</label>
                        <input type="password" name="binance_secret_key" class="form-control" 
                               value="<?php echo sanitize($binanceSecretKey ?? ''); ?>" 
                               placeholder="Enter your Binance secret key">
                        <small class="form-text text-muted">Keep this secret! Never share it.</small>
                    </div>
                    <button type="submit" name="save_binance_credentials" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Binance Credentials
                    </button>
                </form>
            </div>
        </div>

        <!-- Bridge Account Configuration -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-bank"></i> Bridge Account Configuration</h5>
            </div>
            <div class="card-body">
                <?php if ($bridgeAccountDetails && !isset($bridgeAccountDetails['error'])): ?>
                    <div class="alert alert-success mb-3">
                        <i class="bi bi-check-circle"></i> Bridge account is configured and verified.
                        <br><small>Account Type: <?php echo sanitize($bridgeAccountDetails['payment_rail'] ?? 'N/A'); ?></small>
                    </div>
                <?php endif; ?>
                
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">Bridge Customer ID</label>
                        <input type="text" name="bridge_customer_id" class="form-control" 
                               value="<?php echo sanitize($bridgeCustomerId ?? ''); ?>" 
                               placeholder="cus_xxxxx">
                        <small class="form-text text-muted">Your Bridge customer ID (starts with "cus_")</small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Bridge External Account ID</label>
                        <input type="text" name="bridge_external_account_id" class="form-control" 
                               value="<?php echo sanitize($bridgeExternalAccountId ?? ''); ?>" 
                               placeholder="ext_xxxxx">
                        <small class="form-text text-muted">The bank account ID you want to use for payments</small>
                    </div>
                    <button type="submit" name="save_bridge_account" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Bridge Account
                    </button>
                </form>
            </div>
        </div>

        <!-- Automation Settings -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-robot"></i> Automation Settings</h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="mb-3">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" name="auto_payment_enabled" 
                                   id="auto_payment_enabled" <?php echo $autoPaymentEnabled ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="auto_payment_enabled">
                                Enable Automatic Payment Processing
                            </label>
                        </div>
                        <small class="form-text text-muted">
                            When enabled, orders will be automatically paid when received
                        </small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Low Balance Threshold (USDC)</label>
                        <input type="number" name="low_balance_threshold" class="form-control" 
                               value="<?php echo $lowBalanceThreshold; ?>" 
                               step="0.01" min="0">
                        <small class="form-text text-muted">
                            You will receive a notification when balance falls below this amount
                        </small>
                    </div>
                    <button type="submit" name="save_automation_settings" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Automation Settings
                    </button>
                </form>
            </div>
        </div>

        <!-- Wallet Configuration -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-wallet2"></i> Wallet Configuration</h5>
            </div>
            <div class="card-body">
                <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle"></i> 
                    <strong>Security Notice:</strong> Private key will be encrypted before storage. 
                    Make sure your server is secure and database is protected.
                </div>
                
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">Wallet Address (Polygon)</label>
                        <input type="text" name="wallet_address" class="form-control" 
                               value="<?php echo sanitize($walletAddress ?? ''); ?>" 
                               placeholder="0x...">
                        <small class="form-text text-muted">Your Polygon wallet address that holds USDC</small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Private Key</label>
                        <input type="password" name="wallet_private_key" class="form-control" 
                               placeholder="Enter private key (will be encrypted)">
                        <?php if ($walletPrivateKeyEncrypted): ?>
                            <small class="form-text text-success">
                                <i class="bi bi-check-circle"></i> Private key is already configured. 
                                Leave blank to keep current, or enter new key to update.
                            </small>
                        <?php else: ?>
                            <small class="form-text text-muted">Enter private key (without 0x prefix). Will be encrypted before storage.</small>
                        <?php endif; ?>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Withdrawal Wallet Address (Polygon)</label>
                        <input type="text" name="withdrawal_wallet_address" class="form-control" 
                               value="<?php echo sanitize($withdrawalWalletAddress ?? ''); ?>" 
                               placeholder="0x...">
                        <small class="form-text text-muted">Polygon wallet address where USDC will be withdrawn after auto-trading USDT to USDC</small>
                    </div>
                    <button type="submit" name="save_wallet_settings" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Wallet Settings
                    </button>
                </form>
            </div>
        </div>

        <!-- Translation Settings -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-translate"></i> Name Translation Settings</h5>
            </div>
            <div class="card-body">
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> 
                    <strong>Name Translation:</strong> Names are automatically translated to English before sending to Bridge API. 
                    This ensures compatibility with Bridge's requirements.
                </div>
                
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">SYSTRAN API Key</label>
                        <input type="password" name="systran_api_key" class="form-control" 
                               value="<?php echo sanitize($systranApiKey ?? ''); ?>" 
                               placeholder="Enter SYSTRAN API key">
                        <?php if (!empty($systranApiKey)): ?>
                            <small class="form-text text-success">
                                <i class="bi bi-check-circle"></i> SYSTRAN API key is configured. 
                                Name translation is enabled.
                            </small>
                        <?php else: ?>
                            <small class="form-text text-muted">
                                Get your API key from <a href="https://www.systransoft.com/" target="_blank">SYSTRAN</a>. 
                                Required for translating non-English names to English before sending to Bridge.
                            </small>
                        <?php endif; ?>
                    </div>
                    <button type="submit" name="save_translation_settings" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Translation Settings
                    </button>
                </form>
            </div>
        </div>

        <!-- System Information -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> System Information</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <strong>Order Monitor Status:</strong>
                        <?php
                        // Check if cron is running (check last execution time)
                        $lastRun = $settings->get('last_order_check', 'Never');
                        echo $lastRun === 'Never' ? '<span class="badge bg-warning">Not Running</span>' : '<span class="badge bg-success">Running</span>';
                        ?>
                        <br><small class="text-muted">Last check: <?php echo $lastRun; ?></small>
                    </div>
                    <div class="col-md-6">
                        <strong>Database:</strong> 
                        <span class="badge bg-success">Connected</span>
                        <br><small class="text-muted"><?php echo DB_NAME; ?></small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

