<?php
/**
 * Test WebSocket Client
 * Simple test to verify WebSocket client is working
 */

// CRITICAL: Suppress ALL output BEFORE anything else (CLI only)
if (php_sapi_name() === 'cli') {
    // Disable all output immediately - MUST be first
    ini_set('display_errors', '0');
    ini_set('log_errors', '1');
    ini_set('output_buffering', '0');
    ini_set('session.auto_start', '0');
    
    // Clean any existing output buffers
    while (ob_get_level() > 0) {
        ob_end_clean();
    }
    
    // Start fresh buffer to catch ANY stray output
    ob_start();
    
    // Prevent any headers from being sent
    if (function_exists('header_remove')) {
        header_remove();
    }
}

$baseDir = dirname(__DIR__);

// Clear buffer after loading config
if (php_sapi_name() === 'cli' && ob_get_level() > 0) {
    ob_end_clean();
    ob_start();
}

require_once $baseDir . '/includes/config.php';
require_once $baseDir . '/includes/binance-api.php';
require_once $baseDir . '/includes/binance-settings.php';
require_once $baseDir . '/includes/websocket-client.php';

// Clear buffer again after includes
if (php_sapi_name() === 'cli' && ob_get_level() > 0) {
    ob_end_clean();
    ob_start();
}

error_log("=== Testing WebSocket Client ===");

try {
    // Check if Ratchet/Pawl is available
    if (!class_exists('Ratchet\Client\Connector')) {
        throw new Exception("Ratchet/Pawl not found. Run: composer require ratchet/pawl");
    }
    
    error_log("✓ Ratchet/Pawl library found");
    
    $settings = new BinanceP2PSettings($pdo);
    $apiKey = $settings->getBinanceApiKey();
    $secretKey = $settings->getBinanceSecretKey();
    
    if (empty($apiKey) || empty($secretKey)) {
        throw new Exception("Binance API credentials not configured");
    }
    
    error_log("✓ Binance API credentials found");
    
    $binance = new BinanceP2PAPI($apiKey, $secretKey);
    $credentials = $binance->getChatCredentials();
    
    if (isset($credentials['error'])) {
        throw new Exception("Error getting chat credentials: " . $credentials['error']);
    }
    
    $wssUrl = $credentials['data']['chatWssUrl'] ?? null;
    $listenKey = $credentials['data']['listenKey'] ?? null;
    $listenToken = $credentials['data']['listenToken'] ?? null;
    
    if (empty($wssUrl) || empty($listenKey) || empty($listenToken)) {
        throw new Exception("Invalid chat credentials");
    }
    
    error_log("✓ Chat credentials retrieved");
    error_log("  WSS URL: $wssUrl");
    error_log("  Listen Key: $listenKey");
    
    // Test WebSocket client instantiation (without connecting)
    try {
        $wsClient = new BinanceWebSocketClient($wssUrl, $listenKey, $listenToken);
        error_log("✓ WebSocket client instantiated successfully");
    } catch (Exception $e) {
        throw new Exception("Failed to instantiate WebSocket client: " . $e->getMessage());
    }
    
    error_log("✓ WebSocket client is ready to use!");
    error_log("=== Test Complete ===");
    
} catch (Exception $e) {
    error_log("ERROR: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    exit(1);
}

// Clean output buffer before exit
if (php_sapi_name() === 'cli') {
    if (ob_get_level() > 0) {
        ob_end_clean();
    }
    exit(0);
}

